<?php

namespace SayHello\GitInstaller;

class Helpers
{
    public static string $authAdmin = 'administrator';
    private static array $activeNotifications = [];

    public static function checkAuth(): bool
    {
        return current_user_can(self::$authAdmin);
    }

    public static function getPages(): array
    {
        $pages = [];
        foreach (get_pages() as $post) {
            $pages[$post->ID] = get_the_title($post);
        }

        return $pages;
    }

    public static function isEmail($string)
    {
        return filter_var($string, FILTER_VALIDATE_EMAIL);
    }

    public static function isUrl($string)
    {
        return filter_var($string, FILTER_VALIDATE_URL);
    }

    public static function getRestJson($url, $args = [])
    {
        $resp = self::fetchPlainText($url, $args);
        if (is_wp_error($resp)) {
            return $resp;
        }

        $json = json_decode($resp, true);

        if (!$json) return new \WP_Error('json_parse_error', sprintf(
                __('Request to %s could not be processed', 'shgi'),
                '<code>' . $url . '</code>')
        );

        return $json;
    }

    public static function fetchPlainText($url, $args = [])
    {
        $request = wp_remote_get($url, $args);
        if (is_wp_error($request)) return $request;

        $code = wp_remote_retrieve_response_code($request);
        if ($code >= 300) return new \WP_Error('remote_get_error', sprintf(
                __('Invalid request to %s', 'shgi'),
                '<code>' . $url . '</code>')
        );

        $body = wp_remote_retrieve_body($request);

        return str_replace('<?php', '', $body);
    }

    public static function getContentFolder($url = false): string
    {
        $folder = 'shgi';
        $uploadDir = wp_get_upload_dir();
        $baseUrl = trailingslashit($uploadDir['baseurl']);
        $baseDir = trailingslashit($uploadDir['basedir']);
        if (!is_dir($baseDir . $folder . '/')) {
            mkdir($baseDir . $folder . '/');
        }

        if ($url) {
            return $baseUrl . $folder . '/';
        }

        return $baseDir . $folder . '/';
    }

    public static function getTempDir($dir = 'temp', $url = false, $htaccessDenyAll = false): string
    {
        $dir = untrailingslashit($dir);
        $tempDir = Helpers::getContentFolder($url) . $dir . '/';
        if (!is_dir($tempDir)) mkdir($tempDir);
        if ($htaccessDenyAll) {
            if (!file_exists($tempDir . '.htaccess')) {
                file_put_contents($tempDir . '.htaccess', "order deny,allow\ndeny from all");
            }
        }

        return $tempDir;
    }

    public static function checkForFunction($func, $notification = true): bool
    {
        if (!function_exists($func)) {
            $message = 'The function <code>' . $func . '()</code> is not available. Some Parts of <b>' . sayhelloGitInstaller()->name . '</b> won\'t work as expected.';
            if ($notification) {
                self::showAdminNotification($message);
            }
            return false;
        }

        return true;
    }

    public static function showAdminNotification($message, $type = 'error')
    {
        $key = md5("{$type}: {$message}");
        if (!in_array($key, self::$activeNotifications)) {
            add_action('admin_notices', function () use ($message, $type) {
                $class = "notice notice-{$type}";
                printf('<div class="%1$s"><p>%2$s</p></div>', $class, $message);
            });
            self::$activeNotifications[] = $key;
        }
    }

    public static function print($e)
    {
        echo '<pre style="margin-left: 200px">';
        print_r($e);
        echo '</pre>';
    }

    public static function useMustUsePlugins()
    {
        return apply_filters('shgi/Repositories/MustUsePlugins', false);
    }

    public static function sanitizeDir($dir): string
    {
        if (!$dir) return '';
        return trailingslashit($dir);
    }

    public static function addLog($log, string $name = 'debug'): void
    {
        $dir = trailingslashit(self::getTempDir('log', false, true));
        $logString = '[' . date("D Y-m-d H:i:s") . ']' . PHP_EOL;
        $logString .= is_string($log) ? $log : json_encode($log);
        $logString .= PHP_EOL . "-------------------------" . PHP_EOL;
        $file = $dir . apply_filters('post_name', $name) . '_' . date("j.n.Y") . '.log';
        if (!file_exists($file)) file_put_contents($file, '');

        $oldContent = file_get_contents($file);

        file_put_contents($file, $oldContent . $logString);
    }

    public static function sanitizeRepositoryDir($key)
    {
        $key = str_replace('.git', '', $key);
        $key = preg_replace("/[^A-Za-z0-9]/", '-', $key);
        $key = preg_replace('/-+/', '-', $key);
        $key = strtolower($key);
        return $key;
    }
}
